#include <stdio.h>
#include <stdlib.h>
#include <time.h>

/**
 * is_prime - Teste si un entier n est premier (test simple)
 * @n: le nombre à tester
 * 
 * Retourne 1 si premier, 0 sinon
 * 
 * Méthode : on teste la divisibilité par tous les entiers impairs de 3 à sqrt(n)
 */
int is_prime(int n) {
    if (n < 2) return 0;       // Les nombres < 2 ne sont pas premiers
    if (n == 2) return 1;      // 2 est premier
    if (n % 2 == 0) return 0;  // Les nombres pairs > 2 ne sont pas premiers

    for (int i = 3; i * i <= n; i += 2) {
        if (n % i == 0) return 0; // Divisible par i -> pas premier
    }
    return 1; // Aucun diviseur trouvé -> premier
}

/**
 * random_prime - Génère un nombre premier impair aléatoire dans [min..max]
 * 
 * Boucle jusqu'à obtenir un nombre premier
 */
int random_prime(int min, int max) {
    int p;
    do {
        p = rand() % (max - min + 1) + min; // Tirage aléatoire dans l'intervalle
        if (p % 2 == 0) p++;                // S'assurer que p est impair
    } while (!is_prime(p));                 // Répéter tant que p n'est pas premier
    return p;
}

/**
 * egcd - Algorithme d'Euclide étendu pour calculer le PGCD de a et b
 *       et trouver les coefficients x et y tels que a*x + b*y = gcd(a, b)
 * 
 * Retourne le PGCD et remplit x et y par les coefficients de Bézout
 */
int egcd(int a, int b, int *x, int *y) {
    if (b == 0) {
        *x = 1; *y = 0;
        return a; // PGCD trouvé = a
    }
    int x1, y1;
    int gcd = egcd(b, a % b, &x1, &y1);  // Appel récursif
    *x = y1;
    *y = x1 - (a / b) * y1;               // Mise à jour des coefficients
    return gcd;
}

/**
 * modinv - Calcul de l'inverse modulaire de e modulo phi
 *          C'est-à-dire trouver d tel que (d*e) mod phi = 1
 * 
 * Utilise l'algorithme d'Euclide étendu
 * Retourne -1 si l'inverse n'existe pas
 */
int modinv(int e, int phi) {
    int x, y;
    int g = egcd(e, phi, &x, &y);
    if (g != 1) return -1; // Inverse modulaire n'existe que si e et phi sont premiers entre eux
    return (x % phi + phi) % phi; // On normalise x pour qu'il soit positif
}

int main() {
    // Initialisation du générateur de nombres aléatoires avec l'heure actuelle
    srand(time(NULL));

    // Génération aléatoire de deux nombres premiers p et q distincts dans [50..100]
    int p = random_prime(50, 100);
    int q;
    do {
        q = random_prime(50, 100);
    } while (q == p); // On s'assure que p et q soient différents

    int n = p * q;           // Calcul de n = p*q, utilisé pour la clé publique et privée
    int phi = (p - 1) * (q - 1); // Calcul de phi(n), la fonction d'Euler

    // Choix de e : on cherche un nombre impair e premier avec phi
    int e;
    for (e = 3; e < phi; e += 2) {
        int x, y;
        if (egcd(e, phi, &x, &y) == 1)
            break; // e trouvé, premier avec phi
    }

    // Calcul de d, l'inverse modulaire de e modulo phi
    int d = modinv(e, phi);
    if (d == -1) {
        fprintf(stderr, "Erreur : pas d'inverse modulaire pour e=%d\n", e);
        return EXIT_FAILURE;
    }

    // Affichage des résultats (clés publique et privée)
    printf("p = %d, q = %d\n", p, q);
    printf("n = %d\n", n);
    printf("phi = %d\n", phi);
    printf("Clé publique (e, n) : (%d, %d)\n", e, n);
    printf("Clé privée  (d, n) : (%d, %d)\n", d, n);

    return EXIT_SUCCESS;
}

